% Make a DBN for the exocytosis model with the following variables
%
% Time-dependent variables
%
% Coupling variables
% Gex.C, Gcell.C, Scell.C
%
% Observed variables
% The node 'GLP1.obs' corresponds to the 'GLP1.GL' in the GL data model
%
% Time-invariant variables
%
% Parameters
%
% To generate a conditional gaussian model
%
% All variables display gaussian distributions.

% To generate a conditional gaussian model
function [dbn_factory]= make_signaling_dbn(G_mean_signaling, ATP_mean_signaling, GLP1_mean_signaling,...
                                           GLP1R_mean_signaling, cAMP_mean_signaling, Ca_mean_signaling,...
                                           S_mean_signaling, ...
                                           G_cov_signaling, ATP_cov_signaling, GLP1_cov_signaling, ...
                                           GLP1R_cov_signaling, cAMP_cov_signaling, Ca_cov_signaling, ...
                                           S_cov_signaling, cov_scale_signaling,...
                                           alpha_signaling, kATP_signaling, kGLP1_signaling,...
                                           beta_signaling, gamma_signaling, kcAMP_signaling,...
                                           delta_signaling, kCa_signaling, epsilon_signaling);
    
    node_names=  {'G.signaling','Gcell.C','ATP.signaling','GLP1.signaling','GLP1.C',...
                  'GLP1.obs','GLP1R.signaling','GLP1R.C','cAMP.signaling','Ca.signaling',...
                  'S.signaling','Scell.C'}; 
    n= length(node_names);
    
    % Intra - in one time slice
    edges_intra= {'Gcell.C','G.signaling';'GLP1.signaling','GLP1.C';'GLP1.C','GLP1.obs';...
                  'GLP1R.C','GLP1R.signaling';'S.signaling','Scell.C'};
    
    % Inter - between time slices
    edges_inter= {'G.signaling', 'G.signaling'; 'G.signaling','ATP.signaling';'ATP.signaling','ATP.signaling';...
                  'ATP.signaling','cAMP.signaling';'GLP1.signaling','GLP1.signaling';'GLP1.signaling','GLP1R.signaling';...
                  'GLP1R.signaling', 'GLP1R.signaling';'GLP1R.signaling','cAMP.signaling';'cAMP.signaling', 'cAMP.signaling';...
                  'cAMP.signaling', 'Ca.signaling';'Ca.signaling','Ca.signaling'; 'Ca.signaling','S.signaling';'S.signaling','S.signaling'}; 
    
    % 'Equivalence classes' specify how the template is initiated and rolled
    % Specify which CPD is associates with each node in either time
    % slice 1 (eclass1) or in slice 2 onwards (eclass2)
    eclass1_map= containers.Map();
    eclass2_map= containers.Map();
    for i=1:numel(node_names)
        node_name= node_names{i};
        cpd_name= [ node_name '.intra' ];
        eclass1_map(node_name) = cpd_name;
        eclass2_map(node_name) = cpd_name;
    end
    eclass2_map('G.signaling')= 'G.signaling.inter';
    eclass2_map('ATP.signaling')= 'ATP.signaling.inter';
    eclass2_map('GLP1.signaling')= 'GLP1.signaling.inter';   
    eclass2_map('GLP1R.signaling')= 'GLP1R.signaling.inter';   
    eclass2_map('cAMP.signaling')= 'cAMP.signaling.inter';   
    eclass2_map('Ca.signaling')= 'Ca.signaling.inter';    
    eclass2_map('S.signaling')= 'S.signaling.inter';   
    
    % elcass1 (time-slice 0 or all parents are in the same time slice)
    CPDFactories= {};
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'Gcell.C', 0, ...
        {'mean', G_mean_signaling, 'cov', G_cov_signaling} ); % Gcell.C = 1.0 * G
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'G.signaling', 0, ...
        {'mean', 0.0, 'cov', G_cov_signaling*cov_scale_signaling,   'weights', 1.0} ); % G
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'ATP.signaling', 0, ...
        {'mean', ATP_mean_signaling, 'cov', ATP_cov_signaling}); % ATP
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'GLP1.signaling', 0, ...
        {'mean', GLP1_mean_signaling, 'cov', GLP1_cov_signaling}); % GLP1
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'GLP1.C', 0, ...
        {'mean', 0, 'cov', GLP1_cov_signaling*cov_scale_signaling,   'weights', 1.0}); % GLP1
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'GLP1.obs', 0, ...
        {'mean', 0, 'cov', GLP1_cov_signaling*cov_scale_signaling,   'weights', 1.0}); % GLP1
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'GLP1R.C', 0, ...
        {'mean', GLP1R_mean_signaling, 'cov', GLP1R_cov_signaling}); % GLP1R.C
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'GLP1R.signaling', 0, ...
        {'mean', GLP1R_mean_signaling, 'cov', GLP1R_cov_signaling, 'weights', 0.0}); % GLP1R = GLP1R_w_GLP1R_signaling * GLP1R.C
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'cAMP.signaling', 0, ...
        {'mean', cAMP_mean_signaling, 'cov', cAMP_cov_signaling}); % cAMP
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'Ca.signaling', 0, ...
        {'mean', Ca_mean_signaling, 'cov', Ca_cov_signaling}); % Ca2+
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'S.signaling', 0, ...
        {'mean', S_mean_signaling, 'cov', S_cov_signaling}); % S
    
    CPDFactories{end+1} = ...
        CPDFactory('Gaussian_CPD', 'Scell.C', 0, ...
        { 'mean', 0.0,   'cov', S_cov_signaling*cov_scale_signaling,   'weights', 1.0} ); % Scell.C = 1.0 * I

    % eclass2 (time-slice t+1 with parents in the previous time slice)
    % CPD for G(t+1)
    weights_G1_T0= containers.Map(); % parents in slice t
    weights_G1_T1= containers.Map(); % parents in slice t+1
    weights_G1_T0('G.signaling')= 0.0;
    weights_G1_T1('Gcell.C')= 1.0;
    CPDFactories{end+1} = ...
       CPDFactory('Gaussian_CPD', 'G.signaling', 1, ...
        {'mean', 0.0, 'cov', G_cov_signaling*cov_scale_signaling}, weights_G1_T0, weights_G1_T1);  % G(t+1) = 1.0 * G.C(t+1)

    weights_ATP1_T0= containers.Map(); 
    weights_ATP1_T1= containers.Map(); 
    weights_ATP1_T0('G.signaling')= alpha_signaling;
    weights_ATP1_T0('ATP.signaling')= kATP_signaling;
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'ATP.signaling', 1, ...
        {'mean', 0.0, 'cov', ATP_cov_signaling*cov_scale_signaling}, weights_ATP1_T0, weights_ATP1_T1);  % ATP(t+1) = kATP_signaling * ATP(t) + alpha_signaling * G(t)
    
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'GLP1.signaling', 1, ...
        {'mean', 0.0, 'cov', GLP1_cov_signaling*cov_scale_signaling, 'weights', 1.0}); % GLP1(t+1) = 1.0 * GLP1(t)
    
    weights_GLP1R1_T0= containers.Map();
    weights_GLP1R1_T1= containers.Map(); 
    weights_GLP1R1_T0('GLP1.signaling')= kGLP1_signaling;
    weights_GLP1R1_T0('GLP1R.signaling')= 0.0;
    weights_GLP1R1_T1('GLP1R.C')= 0.0;
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'GLP1R.signaling', 1, ...
        {'mean', 0.0, 'cov', GLP1R_cov_signaling*cov_scale_signaling}, weights_GLP1R1_T0, weights_GLP1R1_T1); % GLP1R(t+1) = 0.0 * GLP1R(t) + 0.0 * GLP1R.C(t) + kGLP1_signaling * GLP1(t)
    
    weights_cAMP1_T0= containers.Map(); 
    weights_cAMP1_T1= containers.Map(); 
    weights_cAMP1_T0('ATP.signaling')= beta_signaling;
    weights_cAMP1_T0('GLP1R.signaling')= gamma_signaling;
    weights_cAMP1_T0('cAMP.signaling')= kcAMP_signaling;
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'cAMP.signaling', 1, ...
        {'mean', 0.0, 'cov', cAMP_cov_signaling*cov_scale_signaling}, weights_cAMP1_T0, weights_cAMP1_T1);  % cAMP(t+1) = kcAMP_signaling* cAMP(t) + gamma_signaling*GLP1R(t) + ATP_w_cAMP_signaling* ATP(t)
    
    weights_Ca1_T0= containers.Map(); 
    weights_Ca1_T1= containers.Map(); 
    weights_Ca1_T0('cAMP.signaling')= delta_signaling;
    weights_Ca1_T0('Ca.signaling')= kCa_signaling;
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'Ca.signaling', 1, ...
        {'mean', 0.0, 'cov', Ca_cov_signaling*cov_scale_signaling}, weights_Ca1_T0, weights_Ca1_T1); % Ca(t+1) = kCa_signaling * Ca(t) + deltas_signaling * cAMP(t)

    weights_S1_T0= containers.Map(); 
    weights_S1_T1= containers.Map(); 
    weights_S1_T0('Ca.signaling')= epsilon_signaling;
    weights_S1_T0('S.signaling')= 0.0;
    CPDFactories{end+1}=  ...
        CPDFactory('Gaussian_CPD', 'S.signaling', 1, ...
        {'mean', 0.0, 'cov', S_cov_signaling*cov_scale_signaling}, weights_S1_T0, weights_S1_T1); % S(t+1) = 0.0 * S(t) + epsilon_signaling * Ca(t) 

    % Final DBN factory
    dbn_factory= DBNFactory( ...
        node_names, edges_intra, edges_inter, ...
        eclass1_map, eclass2_map, CPDFactories);
  end

